<?php
session_start();
require_once 'includes/config.php'; 

// 1. DETERMINAR EL NIVEL DE ACCESO ACTUAL
// Si la sesión no está establecida o no es 'empleado', usa el nivel por defecto 'publico'.
$user_level = $_SESSION['user_level'] ?? DEFAULT_USER_LEVEL;
error_log("DEBUG: Nivel de acceso determinado: '" . $user_level . "'");

// Parámetros de Similitud y Respuesta por Defecto
// Estas constantes ya están definidas en config.php, no es necesario redefinirlas.
/*
define('UMBRAL_SIMILITUD_LIKE', 0.6);
define('UMBRAL_SIMILITUD_LEVENSHTEIN', 0.5);
define('ID_RESPUESTA_NO_ENTIENDO', 1);
*/

function normalizarTexto($texto) {
    $texto = mb_strtolower($texto, 'UTF-8');
    
    $texto = str_replace(
        ['á', 'é', 'í', 'ó', 'ú', 'ñ', 'ü'],
        ['a', 'e', 'i', 'o', 'u', 'n', 'u'],
        $texto
    );
    $texto = trim($texto);
    $texto = preg_replace('/\\s+/', ' ', $texto);
    return $texto;
}

try {
    // 2. Usar las constantes para la conexión PDO
    $dsn = "mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME . ";charset=utf8mb4"; // Usamos utf8mb4 para consistencia
    $conn = new PDO($dsn, DB_USER, DB_PASS);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $conn->exec("SET NAMES 'utf8mb4'"); // Refuerzo de codificación

    $pregunta_usuario = normalizarTexto($_POST['pregunta'] ?? ''); // Uso de '??' para evitar error si no hay 'pregunta'
    if (empty($pregunta_usuario)) {
        throw new Exception("Consulta de usuario vacía.");
    }
    error_log("DEBUG: Pregunta del usuario normalizada: '" . $pregunta_usuario . "'");

    $mejor_id = null;
    $respuesta_a_devolver = "";
    $opciones_json_string = "[]";

    // 3. DEFINICIÓN DEL FILTRO DE ACCESO
    // Si es 'empleado', puede ver todas las respuestas (publico O empleado).
    $acceso_filtro = "acceso_nivel = 'publico'";
    if ($user_level === 'empleado') {
        $acceso_filtro = "(acceso_nivel = 'publico' OR acceso_nivel = 'empleado')";
    }
    
    // Consulta LIKE con filtro de acceso
    $query_like = "SELECT id, respuesta, pregunta, opciones_respuesta FROM chatbot_data WHERE pregunta LIKE :pregunta AND $acceso_filtro";
    $stmt_like = $conn->prepare($query_like);
    $pregunta_like_param = "%" . $pregunta_usuario . "%";
    $stmt_like->bindParam(':pregunta', $pregunta_like_param, PDO::PARAM_STR);
    $stmt_like->execute();
    $resultados_like = $stmt_like->fetchAll(PDO::FETCH_ASSOC);

    error_log("DEBUG: Se encontraron " . count($resultados_like) . " resultados potenciales con LIKE (Filtro de acceso: $user_level).");

    $max_similitud_like_encontrada = 0;
    $mejor_resultado_like_fila = null;

    if (!empty($resultados_like)) {
        foreach ($resultados_like as $fila_like) {
            
            $palabras_clave_db = explode(',', $fila_like['pregunta']);
            foreach ($palabras_clave_db as $pc_db) {
                $pc_db_normalizada = normalizarTexto($pc_db);
                $similitud_actual = calcularSimilitud($pregunta_usuario, $pc_db_normalizada);
                error_log("DEBUG: [LIKE] Comparando '" . $pregunta_usuario . "' con palabra clave DB '" . $pc_db_normalizada . "'. Similitud: " . ($similitud_actual * 100) . "% (ID: " . $fila_like['id'] . ")");

                if ($similitud_actual > $max_similitud_like_encontrada) {
                    $max_similitud_like_encontrada = $similitud_actual;
                    $mejor_resultado_like_fila = $fila_like;
                }
            }
        }
    }

    if ($mejor_resultado_like_fila && $max_similitud_like_encontrada >= UMBRAL_SIMILITUD_LIKE) {
        $mejor_id = $mejor_resultado_like_fila['id'];
        error_log("DEBUG: Coincidencia prioritaria seleccionada por LIKE. ID: " . $mejor_id . ", Similitud: " . ($max_similitud_like_encontrada * 100) . "%.");

    } else {
       error_log("DEBUG: No se encontró coincidencia LIKE con umbral o no hay resultados. Intentando Levenshtein con todas las entradas permitidas.");

        // Consulta ALL con filtro de acceso
        $query_all = "SELECT id, respuesta, pregunta, opciones_respuesta FROM chatbot_data WHERE $acceso_filtro";
        $stmt_all = $conn->query($query_all);
        $preguntas_db_todas = $stmt_all->fetchAll(PDO::FETCH_ASSOC);

        $max_similitud_levenshtein = 0;
        $mejor_id_levenshtein = null;

        foreach ($preguntas_db_todas as $fila_levenshtein) {
            $palabras_clave_db = explode(',', $fila_levenshtein['pregunta']);
            foreach ($palabras_clave_db as $pc_db) {
                $pc_db_normalizada = normalizarTexto($pc_db);
                $similitud = calcularSimilitud($pregunta_usuario, $pc_db_normalizada);
                error_log("DEBUG: [LEVENSHTEIN] Comparando '" . $pregunta_usuario . "' con palabra clave DB '" . $pc_db_normalizada . "'. Similitud: " . ($similitud * 100) . "% (ID: " . $fila_levenshtein['id'] . ")");

                if ($similitud > $max_similitud_levenshtein) {
                    $max_similitud_levenshtein = $similitud;
                    $mejor_id_levenshtein = $fila_levenshtein['id'];
                }
            }
        }

        if ($mejor_id_levenshtein && $max_similitud_levenshtein >= UMBRAL_SIMILITUD_LEVENSHTEIN) {
            $mejor_id = $mejor_id_levenshtein;
            error_log("DEBUG: Mejor similitud Levenshtein final encontrada: " . ($max_similitud_levenshtein * 100) . "% para ID: " . $mejor_id . ".");
        } else {
            $mejor_id = ID_RESPUESTA_NO_ENTIENDO;
            error_log("DEBUG: Similitud (LIKE y Levenshtein) bajo el umbral. Cayendo a ID por defecto: " . $mejor_id . ".");
        }
    }

    // Recuperación de la respuesta final (sin filtro, ya que el ID ya fue filtrado)
    if ($mejor_id !== null) {
        $stmt_final = $conn->prepare("SELECT respuesta, opciones_respuesta FROM chatbot_data WHERE id = :id");
        $stmt_final->bindParam(':id', $mejor_id, PDO::PARAM_INT);
        $stmt_final->execute();
        $fila_final = $stmt_final->fetch(PDO::FETCH_ASSOC);

           error_log("DEBUG: Datos recuperados para la ID final (" . $mejor_id . "): " . json_encode($fila_final));

        if ($fila_final) {
            $respuesta_a_devolver = $fila_final['respuesta'];
            $opciones_json_string = $fila_final['opciones_respuesta'];
        } else {
            $respuesta_a_devolver = "Error: No se pudo recuperar la respuesta para la ID " . $mejor_id . ".";
            $opciones_json_string = "[]";
            error_log("ERROR: No se encontraron datos en la DB para la ID final: " . $mejor_id);
        }
    } else {
        $respuesta_a_devolver = "Error interno: No se pudo determinar una ID de respuesta.";
        $opciones_json_string = "[]";
        error_log("ERROR: La variable \$mejor_id es null al final del proceso.");
    }

    $opciones_array = [];
    if (!empty($opciones_json_string)) {
        $temp_opciones_array = json_decode($opciones_json_string, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($temp_opciones_array)) {
            $opciones_array = $temp_opciones_array;
            error_log("DEBUG: Opciones decodificadas como JSON.");
        } else {
            error_log("DEBUG: Opciones no son JSON válido, intentando formato 'tipo:valor,tipo:valor'.");
            $opciones_parts = explode(',', trim($opciones_json_string, '[]'));
            foreach ($opciones_parts as $part) {
                $sub_parts = explode(':', $part, 2);
                if (count($sub_parts) === 2) {
                    $opciones_array[] = ['tipo' => trim($sub_parts[0]), 'valor' => trim($sub_parts[1])];
                }
            }
        }
    }
    error_log("DEBUG: Opciones finales para el frontend: " . json_encode($opciones_array));

    header('Content-Type: application/json');
    echo json_encode(['respuesta' => $respuesta_a_devolver, 'opciones' => $opciones_array]);

} catch (PDOException $e) {
    error_log("ERROR: PDO Exception: " . $e->getMessage());
    header('Content-Type: application/json');
    echo json_encode(['respuesta' => "Error interno del servidor (PDO): " . $e->getMessage(), 'opciones' => []]);
} catch (Exception $e) {
    error_log("ERROR: General Exception: " . $e->getMessage());
    header('Content-Type: application/json');
    echo json_encode(['respuesta' => "Un error inesperado ha ocurrido: " . $e->getMessage(), 'opciones' => []]);
}

function calcularSimilitud($pregunta1, $pregunta2) {
    if (empty($pregunta1) || empty($pregunta2)) {
        return 0;
    }

    $len1 = mb_strlen($pregunta1, 'UTF-8');
    $len2 = mb_strlen($pregunta2, 'UTF-8');

    $distancia = levenshtein($pregunta1, $pregunta2);

    $max_len = max($len1, $len2);
    if ($max_len == 0) {
        return 1;
    }

    $similitud = 1 - ($distancia / $max_len);
    return $similitud;
}
?>