<?php
// Configuración de la respuesta HTTP y zona horaria
header('Content-Type: application/json');
date_default_timezone_set('America/La_Paz'); 

// 1. Incluir el archivo de configuración existente para las credenciales de la BD
// La ruta es relativa al directorio donde se ejecuta este script
require_once __DIR__ . '/includes/config.php'; 

// 2. Conexión a la Base de Datos (usando PDO)
try {
    // Las constantes DB_SERVER, NOMBRE_DB, DB_USER, DB_PASS vienen de config.php
    $dsn = "mysql:host=" . DB_SERVER . ";dbname=" . NOMBRE_DB . ";charset=utf8mb4";
    $pdo = new PDO($dsn, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    // Si la conexión falla, se devuelve un error 500
    http_response_code(500);
    echo json_encode(array("status" => "error", "mensaje" => "Error de conexión interna."));
    exit();
}

// 3. Recepción y Sanitización de Datos (App Inventor envía datos por POST)
// Usamos filter_input para mayor seguridad
$codigo_qr = filter_input(INPUT_POST, 'qr', FILTER_SANITIZE_STRING);
$ci_usuario = filter_input(INPUT_POST, 'ci', FILTER_SANITIZE_STRING);
$proceso = filter_input(INPUT_POST, 'proceso', FILTER_SANITIZE_STRING);
$sucursal_origen = filter_input(INPUT_POST, 'sucursal_origen', FILTER_SANITIZE_STRING);
$descripcion = isset($_POST['descripcion']) ? filter_input(INPUT_POST, 'descripcion', FILTER_SANITIZE_STRING) : null;
$descripcion = filter_input(INPUT_POST, 'descripcion', FILTER_SANITIZE_STRING);

// 4. Validación Mínima de Datos
// La clave aquí es verificar que la APP de App Inventor haya enviado estos datos.
if (empty($codigo_qr) || empty($ci_usuario) || !in_array($proceso, ['ENTRADA', 'SALIDA'])) {
    http_response_code(400); // Bad Request
    echo json_encode(array("status" => "error", "mensaje" => "Faltan datos requeridos (QR, CI o Proceso inválido)."));
    exit();
}

// 5. Inserción de Datos en la Tabla
try {
    // Si la APP no envía la sucursal, asignamos un valor por defecto.
    $sucursal_origen = $sucursal_origen ?: 'Desconocida';
    $sucursal_destino = $sucursal_destino ?: 'N/A';

    $sql = "INSERT INTO movimientos_activos (codigo_qr, descripcion, ci_usuario, proceso, sucursal_origen, sucursal_destino) 
            VALUES (:qr, :desc, :ci, :proc, :origen, :destino)";
            
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':qr' => $codigo_qr,
        ':desc' => $descripcion, // Se envía nulo si no se incluye en la APP
        ':ci' => $ci_usuario,
        ':proc' => $proceso,
        ':origen' => $sucursal_origen,
        ':destino' => $sucursal_destino
    ]);

    // 6. Respuesta Exitosa para App Inventor
    http_response_code(200);
    echo json_encode(array(
        "status" => "ok",
        "mensaje" => "Movimiento registrado con éxito.",
        "id_registro" => $pdo->lastInsertId()
    ));

} catch (PDOException $e) {
    http_response_code(500);
    // Error en la consulta SQL (por ejemplo, la tabla no existe o faltan columnas)
    echo json_encode(array("status" => "error", "mensaje" => "Error al guardar el movimiento."));
}
?>